package com.hero.objects;

import java.util.ArrayList;

import javax.swing.JPanel;

import org.jdom.Element;

import com.hero.HeroDesigner;
import com.hero.Rules;
import com.hero.objects.modifiers.Linked;
import com.hero.objects.modifiers.Modifier;
import com.hero.objects.powers.CompoundPower;
import com.hero.ui.dialog.GenericDialog;
import com.hero.ui.dialog.VPPDialog;
import com.hero.ui.widgets.PopupMessage;
import com.hero.util.Rounder;
import com.hero.util.XMLUtility;

/**
 * Copyright (c) 2000 - 2005, CompNet Design, Inc. All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, is prohibited unless the following conditions are met: 1.
 * Express written consent of CompNet Design, Inc. is obtained by the developer.
 * 2. Redistributions must retain this copyright notice. THIS SOFTWARE IS
 * PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * @author CompNet Design, Inc.
 * @version $Revision$
 */

public class VariablePowerPool extends List {

	protected String xmlID = "VPP";

	public VariablePowerPool(Element root) {
		super(root);
		if (HeroDesigner.getActiveTemplate().is6E()) {
			levelValue = 1;
			levelCost = 1;
			Adder cc = (Adder) findObjectByID(getAvailableAdders(),
					"CONTROLCOST");
			if (cc == null) {
				Adder ad = new Adder();
				ad.setDisplay("Control Cost");
				ad.setXMLID("CONTROLCOST");
				ad.setBaseCost(0);
				ad.setDisplayInString(false);
				ad.setExclusive(true);
				ad.setLevelCost(1);
				ad.setLevelValue(2);
				ad.setMinimumLevel(0);
				ad.setMaxLevel(99999);
				ad.setRequired(true);
				ad.setSelected(true);
				getAssignedAdders().add(ad);
				getAvailableAdders().add(ad);
			} else {
				cc.setRequired(true);
			}
		}
	}

	public VariablePowerPool(String name) {
		super(name);
		this.name = "";
		baseCost = 0;
		minimumCost = 1;
		maxCost = 9999;
		levels = 10;
		minimumLevel = 1;
		minSet = true;
		maxSet = false;
		types = new ArrayList<String>();
		types.add("VPP");
		userInput = true;
		otherInputAllowed = true;
		setOptions(new ArrayList<Adder>());
		inputLabel = "Type";
		allowsOtherModifiers = true;
		allowsOtherAdders = true;
		if (HeroDesigner.getActiveTemplate().is6E()) {
			levelValue = 1;
			levelCost = 1;
			Adder cc = (Adder) findObjectByID(getAssignedAdders(),
					"CONTROLCOST");
			if (cc == null) {
				Adder ad = new Adder();
				ad.setDisplay("Control Cost");
				ad.setXMLID("CONTROLCOST");
				ad.setBaseCost(0);
				ad.setDisplayInString(false);
				ad.setExclusive(true);
				ad.setLevelCost(1);
				ad.setLevelValue(2);
				ad.setMinimumLevel(0);
				ad.setMaxLevel(99999);
				ad.setRequired(true);
				ad.setSelected(true);
				getAssignedAdders().add(ad);
				getAvailableAdders().add(ad);
			}
		}
		resetID();
	}

	@Override
	public double getActiveCost() {
		double total = getTotalCost();
		double pool = 0;
		if (getLevelValue() != 0) {
			pool = (getLevels() / getLevelValue()) * getLevelCost();
		}
		double control = pool/2;
		
		if (HeroDesigner.getActiveTemplate().is6E()) {
			Adder c = (Adder) findObjectByID(getAssignedAdders(), "CONTROLCOST");
			if (c != null) {
				control = c.getTotalCost();
			}
		} 
		double advantageTotal = 0d;
		boolean advantagesApplied = false;
		for (int i = 0; i < assignedModifiers.size(); i++) {
			Modifier mod = assignedModifiers.get(i);
			if (mod.getTotalValue() > 0) {
				advantageTotal += mod.getTotalValue();
				advantagesApplied = true;
			}
		}
		for (int i = 0; i < getPrivateMods().size(); i++) {
			Modifier mod = getPrivateMods().get(i);
			if (!mod.isLimitation() && (mod.getTotalValue() > 0)) {
				advantageTotal += mod.getTotalValue();
				advantagesApplied = true;
			}
		}
		double ret = (total-pool) * (1 + advantageTotal);
		ret += pool;
		
		if (advantagesApplied) {
			ret = Rounder.roundHalfDown(ret);
		}
		return ret;
	}

	@Override
	public String getColumn2Output() {
		String ret = getAlias();
		if ((getName() != null) && (getName().trim().length() > 0)) {
			ret = "<i>" + getName() + ":</i>  " + ret;
		}
		if ((getInput() != null) && (getInput().trim().length() > 0)) {
			ret += " (" + getInput() + ")";
		}
		Adder control = (Adder) findObjectByID(getAssignedAdders(),
				"CONTROLCOST");
		long cc = 0;
		if (control != null) {
			cc = control.getLevels();
		} else {
			double pool = 0;
			if (getLevelValue() != 0) {
				pool = (getLevels() / getLevelValue()) * getLevelCost();
			}
			cc = Rounder.roundHalfDown(pool/2);
		}
		ret += ", " + getLevels() + " base + " + cc + " control cost";
		String adderString = getAdderString();
		if (adderString.trim().length() > 0) {
			ret += "; " + adderString;
		}
		ret += getModifierString();
		return ret;
	}

	@Override
	public String getColumn2Prefix(GenericObject child) {
		updateChildPositions();
		int childPosition = getObjects().indexOf(child);
		childPosition++;
		return childPosition + ") ";
	}

	@Override
	public String getColumn2Suffix(GenericObject child) {
		return " Real Cost: "
				+ Rounder.roundHalfDown(child.getRealCostPreList());
	}

	@Override
	public GenericDialog getDialog(boolean isNew, boolean isPower) {
		VPPDialog dialog = new VPPDialog(this, isNew, isPower);
		return dialog;
	}

	@Override
	public String getDisplay() {
		return "Variable Power Pool";
	}

	@Override
	public ArrayList<String> getExamples() {
		examples = new ArrayList<String>();
		examples.add("Gadget Pool");
		examples.add("Magic Pool");
		examples.add("Mimicry Pool");
		return examples;
	}

	@Override
	public double getLevelCost() {
		return 1;
	}

	@Override
	public double getLevelValue() {
		return 1;
	}

	@Override
	public double getRealCostForChild(GenericObject child) {
		return 0;
	}

	@Override
	public double getRealCostPreList() {
		double active = getActiveCost();
		double pool = 0;
		if (getLevelValue() != 0) {
			pool = (getLevels() / getLevelValue()) * getLevelCost();
		}
		double control = pool / 2;
		boolean limitationsApplied = false;
		double limitationTotal = 0d;
		for (int i = 0; i < assignedModifiers.size(); i++) {
			Modifier mod = assignedModifiers.get(i);
			if (mod.getTotalValue() < 0) {
				limitationTotal += mod.getTotalValue();
				limitationsApplied = true;
			}
		}
		for (int i = 0; i < getPrivateMods().size(); i++) {
			Modifier mod = getPrivateMods().get(i);
			if (mod.getTotalValue() < 0) {
				limitationTotal += mod.getTotalValue();
				limitationsApplied = true;
			}
		}
		double ret = (active-pool) / (1d + Math.abs(limitationTotal));
		ret += pool;
		if (limitationsApplied) {
			ret = Rounder.roundHalfDown(ret);
		}

		if (ret < 1) {
			ret = 1;
		}

		if ((ret < 1) && (active > 0)) {
			ret = 1;
		}

		if ((HeroDesigner.getActiveHero() != null)
				&& HeroDesigner.getActiveHero().getRules().multiplierAllowed()
				&& (getMultiplier() != 1)) {
			ret = ret * getMultiplier();
			ret = Rounder.roundHalfDown(ret);
		} else if ((HeroDesigner.getActiveHero() != null)
				&& HeroDesigner.getActiveHero().getRules().multiplierAllowed()
				&& (getParentList() != null)
				&& (getParentList().getMultiplier() != 1)) {
			ret = ret * getParentList().getMultiplier();
			ret = Rounder.roundHalfDown(ret);
		}
		if (getQuantity() > 1) {
			double q = getQuantity();
			int doublings = 0;
			while (q > 1d) {
				doublings += 1;
				q = q / 2d;
			}
			ret += doublings * 5;
		}
		return ret;
	}

	@Override
	public Element getSaveXML() {
		Element root = getGeneralSaveXML();
		root.setName("VPP");
		root.setAttribute("QUANTITY", getQuantity() + "");
		return root;
	}

	@Override
	public double getTotalCost() {
		double total = 0;
		if (HeroDesigner.getActiveTemplate().is6E()) {
			if (getLevelValue() != 0) {
				double bd = (getLevels() / getLevelValue()) * getLevelCost();
				total += Rounder.roundHalfDown(bd);
				Adder c = (Adder) findObjectByID(getAssignedAdders(), "CONTROLCOST");
				if (c != null) {
					total += c.getTotalCost();
				}
			}
			for (int i = 0; i < getAssignedAdders().size(); i++) {
				Adder ad = getAssignedAdders().get(i);
				if (!ad.isRequired()) total += ad.getTotalCost();
			}
			for (int i = 0; i < getPrivateAdders().size(); i++) {
				Adder ad = getPrivateAdders().get(i);
				if (!ad.isRequired()) total += ad.getTotalCost();
			}
		} else {
			if (getLevelValue() != 0) {
				double bd = (getLevels() / getLevelValue()) * getLevelCost();
				total += Rounder.roundHalfDown(bd);
				total += Rounder.roundHalfDown(bd/2);
			}
			for (int i = 0; (i < getAssignedAdders().size())
					&& (getPrivateAdders().size() == 0); i++) {
				Adder ad = getAssignedAdders().get(i);
				total += ad.getTotalCost();
			}
			for (int i = 0; i < getPrivateAdders().size(); i++) {
				Adder ad = getPrivateAdders().get(i);
				total += ad.getTotalCost();
			}
		}

		return total;
	}

	@Override
	public void init(Element root) {
		super.init(root);
		baseCost = 0;
		minimumCost = 1;
		maxCost = 999;
		levels = 10;
		minimumLevel = 1;
		minSet = true;
		maxCost = 999;
		maxSet = false;
		types = new ArrayList<String>();
		types.add("VPP");
		userInput = true;
		otherInputAllowed = true;
		setOptions(new ArrayList<Adder>());
		inputLabel = "Type";
		abbreviation = "VPP";
	}

	@Override
	public boolean objectAllowed(GenericObject o, boolean displayPopup) {
		boolean ret = true;
		if (getAlias().trim().length() == 0) {
			error = "";
			return false;
		}
		if (o instanceof CompoundPower) {
			CompoundPower cp = (CompoundPower) o;
			for (int i = 0; i < cp.getPowers().size(); i++) {
				GenericObject obj = cp.getPowers().get(i);
				boolean check = objectAllowed(obj, displayPopup);
				if (!check) {
					return check;
				}
			}
			// if (cp.getPowers().size()>0) return true;
		}
		if (!o.allowsOtherModifiers() && (getAssignedModifiers().size() > 0)) {
			error = o.getAlias()
					+ " is not allowed to have Modifiers assigned to it in its current configuration.\n\n"
					+ o.getAlias() + " will be placed outside of the list.";
			return false;
		}
		if (o instanceof List) {
			error = "You cannot add a List into a VPP.  New list will be placed outside of selection.";
			return false;
		} else if (o instanceof VariablePowerPool) {
			error = "You cannot add a VPP into another VPP.  The Variable Power Pool will be placed outside of the selection.";
			return false;
		} else if ((getAssignedModifiers().size() > 0)
				&& HeroDesigner.getInstance().getPrefs()
						.isModifierIntelligenceOn()) {
			List oldParent = o.getParentList();
			o.setParent(null);

			ArrayList<Modifier> listMods = getAssignedModifiers();
			ArrayList<Modifier> orig = o.getAssignedModifiers();
			ArrayList<Modifier> slotMods = (ArrayList<Modifier>) orig.clone();
			ArrayList<Modifier> addFromList = new ArrayList<Modifier>();
			for (int i = 0; i < listMods.size(); i++) {
				Modifier mod = listMods.get(i);
				if (GenericObject.findObjectByID(orig, mod.getXMLID()) == null) {
					addFromList.add(mod);
				}
			}
			slotMods.addAll(addFromList);
			o.setAssignedModifiers(slotMods);
			INNER: for (int j = addFromList.size() - 1; j >= 0; j--) {
				Modifier mod = addFromList.get(j);
				slotMods.remove(mod);
				o.setAssignedModifiers(slotMods);
				String check = mod.included(o);
				if (check.trim().length() > 0) {
					error = mod.getAlias() + " cannot be applied to "
							+ o.getAlias() + ".\n\nReason:  " + check + "\n\n"
							+ o.getAlias()
							+ " will be placed outside of the VPP.";
					ret = false;
					break INNER;
				}
				slotMods.add(mod);
			}
			o.setAssignedModifiers(orig);
			o.setParent(oldParent);
		}
		long pool = getLevels();
		if (HeroDesigner.getActiveTemplate().is6E()) {
			Adder control = (Adder) findObjectByID(getAssignedAdders(),
					"CONTROLCOST");
			long cc = 0;
			if (control != null) {
				cc = control.getLevels();
			} else {
				cc = Rounder.roundHalfDown(getRealCost() - getLevels());
			}
			pool = cc;
		}
		long activePool = 0;
		double advantageTotal = 1;
		double limitationTotal = 1;
		for (int i = 0; i < getAssignedModifiers().size(); i++) {
			Modifier mod = getAssignedModifiers().get(i);
			if (mod.getTypes().contains("VPP")) {
				continue;
			}
			if (mod.getTotalValue() > 0) {
				advantageTotal += mod.getTotalValue();
			} else {
				limitationTotal += -1 * mod.getTotalValue();
			}
		}
		pool = Rounder.roundHalfDown(pool * advantageTotal);
		activePool = pool;
		
		// pool = Rounder.roundHalfDown(pool / limitationTotal);
		List oldParent = o.getParentList();
		o.setParent(this);
		if (o.getRealCostPreList() > getLevels()) {
			error = "Real Cost for "
					+ o.getAlias()
					+ " exceeds the available points in the VPP.  Item will be placed outside of the Pool's sample powers list.";
			o.setParent(oldParent);
			return false;
		} else if (o.getActiveCost() > activePool) {
			error = "The Active Cost of " + o.getAlias()
					+ " exceeds the Pool Cost of the VPP.  " + o.getAlias()
					+ " will be placed outside of this VPP.";
			if (HeroDesigner.getActiveTemplate().is6E()) {
				error = "The Active Cost of " + o.getAlias()
						+ " exceeds the Control Cost of the VPP.  " + o.getAlias()
						+ " will be placed outside of this VPP.";
			}
			o.setParent(oldParent);
			ret = false;
		} else if ((GenericObject.findObjectByID(o.getAssignedModifiers(),
				"LINKED") != null)
				&& (HeroDesigner.getActiveHero().getRules()
						.getLinkAcrossFramework() > Rules.IGNORE)) {
			o.setParent(oldParent);
			Linked link = (Linked) GenericObject.findObjectByID(o
					.getAssignedModifiers(), "LINKED");
			GenericObject linkObj = link.getValue();
			if ((linkObj != null)
					&& (linkObj.getParentList() != null)
					&& ((o.getMainPower() == null)
							|| (linkObj.getMainPower() == null) || !o
							.getMainPower().equals(linkObj.getMainPower()))) {
				if ((linkObj.getParentList() instanceof Multipower)
						|| (linkObj.getParentList() instanceof ElementalControl)
						|| (linkObj.getParentList() instanceof VariablePowerPool)) {
					if (HeroDesigner.getActiveHero().getRules()
							.getLinkAcrossFramework() == Rules.DONOTALLOW) {
						error = "You cannot Link two different slots of a Power Framework";
						ret = false;
					} else {
						String msg = "Warning!  You should not Link two different slots of a Power Framework.  Use a Compound Power instead.";
						if (displayPopup) {
							PopupMessage popup = PopupMessage.getInstance(
									HeroDesigner.getAppFrame(),
									(JPanel) HeroDesigner.getAppFrame()
											.getContentPane(), msg, true);
							popup.setVisible(true);
							displayPopup = false;
						}
					}
				}
			}
		}
		if (!ret) {
			o.setParent(oldParent);
			return ret;
		}
		o.setParent(oldParent);
		if (o.getTypes().contains("SPECIAL")
				&& (HeroDesigner.getActiveHero().getRules()
						.getSpecialTypeInFramework() > Rules.IGNORE)) {
			String msg = "Warning!  "
					+ o.getDisplay()
					+ " should not be placed in an Variable Power Pool (or any other Power Framework).";
			if (displayPopup
					&& (HeroDesigner.getActiveHero().getRules()
							.getSpecialTypeInFramework() == Rules.WARN)) {
				PopupMessage popup = PopupMessage.getInstance(HeroDesigner
						.getAppFrame(), (JPanel) HeroDesigner.getAppFrame()
						.getContentPane(), msg, true);
				popup.setVisible(true);
			} else if (HeroDesigner.getActiveHero().getRules()
					.getSpecialTypeInFramework() == Rules.DONOTALLOW) {
				error = msg;
				return false;
			}
			ret = true;
		}
		return ret;
	}

	@Override
	public void restoreFromSave(Element root) {
		if (findObjectByID(getAssignedAdders(), "CONTROLCOST") != null) {
			getAssignedAdders().remove(
					findObjectByID(getAssignedAdders(), "CONTROLCOST"));
		}
		super.restoreFromSave(root);
		for (Adder ad : getAssignedAdders()) {
			System.out.println("FOUND " + ad.getXMLID() + ", levels = "
					+ ad.getLevels() + ", basecost = " + ad.getBaseCost()
					+ ", total cost = " + ad.getTotalCost());
		}
		if (!HeroDesigner.getActiveTemplate().is6E()) {
			availableModifiers = new ArrayList<Modifier>();
		}
		String check = XMLUtility.getValue(root, "QUANTITY");
		if ((check != null) && (check.trim().length() > 0)) {
			try {
				quantity = Integer.parseInt(check);
			} catch (Exception exp) {
				quantity = 1;
			}
		} else {
			quantity = 1;
		}
	}
}